# RAG + LLM + 向量化系统设计文档

## 一、系统目标

构建一个可持续生成长篇小说的系统，满足以下需求：

1. 支持多章节、多人物、多阵营的复杂叙事
2. 保持世界观、人物关系、剧情逻辑一致
3. 每章生成后可向量化存储，用于下一章 RAG 检索
4. 支持章节修改/删除时，向量库同步更新
5. 可控 Prompt 输入，确保写作风格统一

---

## 二、信息层级设计

| 层级        | 内容               | 数据格式                           | 是否 RAG 检索 | 功能              |
| --------- | ---------------- | ------------------------------ | --------- | --------------- |
| L1 世界蓝图   | 世界设定、人物档案、规则、阵营  | JSON                           | ❌ 否       | 提供固定约束，保证逻辑一致性  |
| L2 剧情记忆   | 已生成章节的分块文本       | 向量库（FAISS / Qdrant / Weaviate） | ✅ 是       | 检索与当前章节相关的情节、事件 |
| L3 章节摘要   | 章节标题 + 摘要 + 主要人物 | JSON / Markdown                | ✅ 是       | 检索辅助，缩小上下文范围    |
| L4 上下文桥接  | 上一章摘要 + 结尾 500 字 | Markdown                       | ✅ 是       | 保持衔接自然，情绪与逻辑连贯  |
| L5 当前章节输入 | 标题、摘要、写作指令       | 自然语言                           | ❌ 否       | 明确写作目标与情节点      |

---

## 三、章节生成流程

### Step 1：输入章节目标

* 当前章节标题 + 摘要 + 写作指令
* 系统接收后，准备上下文

### Step 2：RAG 检索上下文

1. **检索剧情记忆层**

   * 根据章节标题、摘要或人物/场景标签检索最相关 chunk
   * 建议 top-K = 5

2. **检索章节摘要层**

   * 辅助判断要引用的前后章节
   * 可选 top-K = 3–5

### Step 3：拼接 Prompt

```
【世界蓝图】（JSON）
{蓝图}

【上一章摘要】
{上一章摘要}

【上一章结尾】
{上一章结尾500字}

【检索到的剧情上下文】（Markdown）
{相关chunk文本拼接}

【当前章节目标】
标题：{chapter_title}
摘要：{chapter_summary}
写作要求：{writing_notes}
```

### Step 4：调用 LLM 生成章节

* 输出章节正文

---

## 四、章节向量化设计

### Step 1：分块策略

* **chunk_size**：300–600 字
* **chunk_overlap**：80–130 字
* **切分逻辑**：

  1. 首选 LangChain `RecursiveCharacterTextSplitter`，按照段落/句子逐级回退切分，自动去除冗余空白
  2. 未安装 LangChain 时退回到内置的段落 + 标点切分策略
* 每块保证语义完整，多句多段落

### Step 2：附加 metadata

* chapter_number
* chapter_title
* chunk_id
* tags（人物、场景、事件，可选）

### Step 3：向量化存储

```python
for chunk in chunks:
    vector_store.upsert({
        "id": unique_id,
        "text": chunk_text,
        "embedding": get_embedding(chunk_text),
        "metadata": {
            "chapter": chapter_number,
            "title": chapter_title,
            "chunk_id": chunk_id,
            "tags": [人物, 场景]
        }
    })
```

---

## 五、章节修改与删除策略

1. **删除操作**

   * 根据 `chapter_number` 或 `chapter_id` 删除对应向量块
   * 避免下一章引用过时内容

```python
vector_store.delete(filter={"chapter": 12})
```

2. **修改操作**

   * 删除旧 chunk
   * 生成新章节
   * 分块向量化并插入数据库
   * 更新章节摘要索引

3. **版本控制（可选）**

   * 每个 chunk 增加 `version` 字段
   * 保留历史版本用于调试或回滚

---

## 六、检索策略

* **上一章摘要 + 结尾**：高权重
* **RAG检索相关 chunk**：中权重
* **蓝图 JSON**：不需检索，直接作为规则约束
* **标签筛选**：人物、场景、事件标签可用于精准检索

---

## 七、Prompt 格式建议

* **蓝图**：JSON
* **检索上下文**：Markdown
* **章节目标**：自然语言
* **系统指令**：固定模板，约束风格与逻辑

---

## 八、数据存储设计

| 数据类型          | 存储形式            | 用途           |
| ------------- | --------------- | ------------ |
| 蓝图 JSON       | 文件 / 数据库        | 人物、世界观、规则约束  |
| 剧情 chunk      | 向量数据库           | RAG 检索，保持上下文 |
| 章节摘要          | JSON / Markdown | 辅助检索与上下文引用   |
| 上一章结尾         | Markdown        | Prompt桥接衔接自然 |
| 标签 / Metadata | 向量库附加字段         | 精准检索         |

---

## 九、扩展优化建议

1. **动态检索 top-K**

   * 章节少 → K 大
   * 章节多 → K 小，保证 token 限制

2. **Chunk 标签化**

   * 每块 chunk 增加人物/场景/事件标签
   * 检索时可加 filter，精确上下文

3. **自动摘要生成**

   * 每章生成后自动提炼摘要
   * 更新章节摘要索引，便于下一章检索

4. **可选多版本管理**

   * 增加 `version` 字段
   * 支持修改回滚
